import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useLocation } from "wouter";
import { FileText, Banknote, Package, MessageSquare, Truck, ArrowUpDown, ArrowUp, ArrowDown } from "lucide-react";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import { Progress } from "@/components/ui/progress";
import { StatusBadge } from "@/components/status-badge";
import type { AllegroOrder, OrderStatus, FulfillmentStatus } from "@shared/schema";

interface OrdersTableProps {
  orders: AllegroOrder[];
  isLoading?: boolean;
  sortBy?: string;
  sortOrder?: "ASC" | "DESC";
  onSort?: (column: string) => void;
}

export function OrdersTable({ 
  orders, 
  isLoading = false,
  sortBy,
  sortOrder,
  onSort
}: OrdersTableProps) {
  const [, navigate] = useLocation();

  // Oblicz maksymalną wartość zamówienia dla paska postępu
  const maxOrderValue = orders.length > 0 
    ? Math.max(...orders.map(o => parseFloat((o as any).summary?.totalToPay?.amount || "0")))
    : 0;

  const SortableHeader = ({ column, children, className = "" }: { column: string; children: React.ReactNode; className?: string }) => {
    const isSorted = sortBy === column;
    return (
      <TableHead 
        className={`cursor-pointer hover-elevate ${className}`}
        onClick={() => onSort?.(column)}
      >
        <div className="flex items-center gap-1">
          {children}
          {isSorted ? (
            sortOrder === 'ASC' ? <ArrowUp className="h-3 w-3" /> : <ArrowDown className="h-3 w-3" />
          ) : (
            <ArrowUpDown className="h-3 w-3 opacity-30" />
          )}
        </div>
      </TableHead>
    );
  };

  if (isLoading) {
    return (
      <div className="rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead className="w-[100px]">Zdjęcia</TableHead>
              <TableHead className="w-[100px]">Numer</TableHead>
              <TableHead className="w-[80px]">Źródło</TableHead>
              <TableHead>Kupujący</TableHead>
              <TableHead className="w-[60px] text-center">
                <Package className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[120px] text-right">Kwota</TableHead>
              <TableHead className="w-[180px]">Wartość</TableHead>
              <TableHead className="w-[140px]">Status płatności</TableHead>
              <TableHead className="w-[150px]">Status realizacji</TableHead>
              <TableHead className="w-[50px] text-center">
                <Banknote className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[50px] text-center">
                <FileText className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[50px] text-center">
                <Truck className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[50px] text-center">
                <MessageSquare className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[140px]">Data</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {[1, 2, 3, 4, 5].map((i) => (
              <TableRow key={i}>
                <TableCell>
                  <div className="h-10 w-10 bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-4 w-32 bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-4 w-24 bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-8 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-right">
                  <div className="h-4 w-20 ml-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-2 w-full bg-muted rounded-full animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-6 w-24 bg-muted rounded-full animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-6 w-28 bg-muted rounded-full animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-4 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-4 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-4 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-4 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-4 w-24 bg-muted rounded animate-pulse" />
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </div>
    );
  }

  if (orders.length === 0) {
    return (
      <div className="rounded-md border p-12 text-center">
        <div className="mx-auto w-24 h-24 mb-4 rounded-full bg-muted flex items-center justify-center">
          <svg
            className="w-12 h-12 text-muted-foreground"
            fill="none"
            viewBox="0 0 24 24"
            stroke="currentColor"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={1.5}
              d="M20 13V6a2 2 0 00-2-2H6a2 2 0 00-2 2v7m16 0v5a2 2 0 01-2 2H6a2 2 0 01-2-2v-5m16 0h-2.586a1 1 0 00-.707.293l-2.414 2.414a1 1 0 01-.707.293h-3.172a1 1 0 01-.707-.293l-2.414-2.414A1 1 0 006.586 13H4"
            />
          </svg>
        </div>
        <h3 className="text-lg font-semibold mb-2">Brak zamówień</h3>
        <p className="text-sm text-muted-foreground max-w-sm mx-auto">
          Nie znaleziono żadnych zamówień. Upewnij się, że połączenie z Allegro jest
          aktywne i zsynchronizuj dane.
        </p>
      </div>
    );
  }

  return (
    <div className="rounded-md border">
      <Table>
        <TableHeader>
          <TableRow>
            <TableHead className="w-[100px]">Zdjęcia</TableHead>
            <SortableHeader column="order_number" className="w-[140px]">Numer</SortableHeader>
            <SortableHeader column="source" className="w-[80px]">Źródło</SortableHeader>
            <SortableHeader column="buyer_login">Kupujący</SortableHeader>
            <TableHead className="w-[60px] text-center">
              <Package className="h-4 w-4 mx-auto" />
            </TableHead>
            <SortableHeader column="total_to_pay_amount" className="w-[120px] text-right">Kwota</SortableHeader>
            <TableHead className="w-[180px]">Wartość</TableHead>
            <SortableHeader column="payment_status" className="w-[140px]">Status płatności</SortableHeader>
            <SortableHeader column="status" className="w-[150px]">Status realizacji</SortableHeader>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Banknote className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Metoda płatności</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <FileText className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Faktura</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Truck className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Przesyłka</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <MessageSquare className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Wiadomość</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <SortableHeader column="order_date" className="w-[140px]">Data</SortableHeader>
          </TableRow>
        </TableHeader>
        <TableBody>
          {orders.map((order) => {
            const o = order as any;
            const buyerFirstName = o.buyer?.firstName || "";
            const buyerLastName = o.buyer?.lastName || "";
            const buyerLogin = o.buyer?.login || "";
            const fullName = buyerFirstName && buyerLastName 
              ? `${buyerFirstName} ${buyerLastName}` 
              : buyerLogin || "—";
            
            const lineItems = o.lineItems || [];
            const hasMessage = o.messageToSeller && o.messageToSeller.trim().length > 0;
            
            return (
              <TableRow
                key={order.id}
                className="hover-elevate cursor-pointer"
                onClick={() => navigate(`/order/${order.orderNumber}`)}
                data-testid={`row-order-${order.orderNumber}`}
              >
                <TableCell>
                  <div className="flex gap-1">
                    {lineItems.slice(0, 3).map((item: any, idx: number) => {
                      const imageFilename = item.imageUrl?.split('/').pop();
                      return (
                        <div key={idx} className="relative">
                          {imageFilename ? (
                            <img 
                              src={`/api/products/images/${imageFilename}`}
                              alt={item.offer?.name || 'Produkt'}
                              className="h-10 w-10 object-cover rounded border"
                              onError={(e) => {
                                (e.target as HTMLImageElement).src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PHJlY3Qgd2lkdGg9IjQwIiBoZWlnaHQ9IjQwIiBmaWxsPSIjZjNmNGY2Ii8+PC9zdmc+';
                              }}
                            />
                          ) : (
                            <div className="h-10 w-10 bg-muted rounded border flex items-center justify-center">
                              <Package className="h-4 w-4 text-muted-foreground" />
                            </div>
                          )}
                        </div>
                      );
                    })}
                    {lineItems.length > 3 && (
                      <div className="h-10 w-10 bg-muted rounded border flex items-center justify-center text-xs font-medium">
                        +{lineItems.length - 3}
                      </div>
                    )}
                  </div>
                </TableCell>
                <TableCell>
                  <div className="flex flex-col">
                    <code
                      className="font-mono text-base font-semibold"
                      data-testid={`text-order-code-${order.allegroOrderId}`}
                    >
                      {order.orderCode || order.allegroOrderId?.substring(0, 8)}
                    </code>
                    <span className="text-xs text-muted-foreground">
                      {format(new Date(o.boughtAt || o.orderDate), "HH:mm", { locale: pl })}
                    </span>
                  </div>
                </TableCell>
                <TableCell>
                  <span className={`inline-flex items-center px-2 py-1 rounded-md text-xs font-medium ${
                    order.source === 'SHOPER' 
                      ? 'bg-blue-500/10 text-blue-600 dark:text-blue-400' 
                      : 'bg-orange-500/10 text-orange-600 dark:text-orange-400'
                  }`}>
                    {order.source || 'ALLEGRO'}
                  </span>
                </TableCell>
                <TableCell>
                  <div>
                    <div
                      className="font-medium"
                      data-testid={`text-buyer-${order.allegroOrderId}`}
                    >
                      {fullName}
                    </div>
                    {buyerLogin && buyerLogin !== fullName && (
                      <div className="text-xs text-muted-foreground">
                        {buyerLogin}
                      </div>
                    )}
                  </div>
                </TableCell>
                <TableCell className="text-center tabular-nums">
                  {lineItems.length || 0}
                </TableCell>
                <TableCell className="text-right font-medium tabular-nums">
                  {(() => {
                    const paidAmount = o.payment?.paidAmount?.amount;
                    const isPaid = paidAmount && parseFloat(paidAmount) > 0;
                    const isCashOnDelivery = o.payment?.type === 'CASH_ON_DELIVERY';
                    const colorClass = isPaid 
                      ? 'text-green-600 dark:text-green-400' 
                      : isCashOnDelivery 
                        ? 'text-orange-600 dark:text-orange-400'
                        : 'text-red-600 dark:text-red-400';
                    return (
                      <span className={colorClass}>
                        {parseFloat(o.summary?.totalToPay?.amount || "0").toFixed(2)} {o.summary?.totalToPay?.currency || "PLN"}
                      </span>
                    );
                  })()}
                </TableCell>
                <TableCell>
                  {(() => {
                    const currentValue = parseFloat(o.summary?.totalToPay?.amount || "0");
                    const percentage = maxOrderValue > 0 ? (currentValue / maxOrderValue) * 100 : 0;
                    return (
                      <div className="space-y-1">
                        <Progress 
                          value={percentage} 
                          className="h-2"
                          data-testid={`progress-order-value-${order.allegroOrderId}`}
                        />
                        <div className="text-xs text-muted-foreground text-center">
                          {percentage.toFixed(0)}%
                        </div>
                      </div>
                    );
                  })()}
                </TableCell>
                <TableCell>
                  <StatusBadge
                    type="payment"
                    status={(() => {
                      const paidAmount = o.payment?.paidAmount?.amount;
                      const isPaid = paidAmount && parseFloat(paidAmount) > 0;
                      const isCashOnDelivery = o.payment?.type === 'CASH_ON_DELIVERY';
                      const refundAmount = parseFloat(o.payment?.refundAmount || "0");
                      const hasRefund = refundAmount > 0;
                      const paidAmountValue = isPaid ? parseFloat(paidAmount) : 0;
                      const netPaidAmount = paidAmountValue - refundAmount;
                      
                      if (hasRefund && netPaidAmount === 0) return 'REFUNDED';
                      if (isPaid) return 'PAID';
                      if (isCashOnDelivery) return 'CASH_ON_DELIVERY';
                      return 'UNPAID';
                    })() as OrderStatus}
                  />
                </TableCell>
                <TableCell>
                  <StatusBadge
                    type="fulfillment"
                    status={(o.status?.toUpperCase() || 'NEW') as FulfillmentStatus}
                  />
                </TableCell>
                <TableCell className="text-center">
                  {o.payment?.type === "CASH_ON_DELIVERY" ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Banknote className="h-4 w-4 mx-auto text-amber-600" data-testid={`icon-payment-cod-${order.allegroOrderId}`} />
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Pobranie</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : o.payment?.type === "ONLINE" ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Banknote className="h-4 w-4 mx-auto text-green-600" data-testid={`icon-payment-online-${order.allegroOrderId}`} />
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Płatność online</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <span className="text-muted-foreground">—</span>
                  )}
                </TableCell>
                <TableCell className="text-center">
                  {o.invoice?.required ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <FileText className="h-4 w-4 mx-auto text-blue-600" data-testid={`icon-invoice-${order.allegroOrderId}`} />
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Z fakturą</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <span className="text-muted-foreground">—</span>
                  )}
                </TableCell>
                <TableCell className="text-center">
                  {(() => {
                    const shipmentList = o.delivery?.shipments || [];
                    const hasTracking = shipmentList.some((s: any) => s.waybill);
                    const trackingNumbers = shipmentList.filter((s: any) => s.waybill).map((s: any) => s.waybill).join(', ');
                    
                    if (hasTracking) {
                      return (
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Truck className="h-4 w-4 mx-auto text-green-600" data-testid={`icon-tracking-${order.allegroOrderId}`} />
                          </TooltipTrigger>
                          <TooltipContent>
                            <p>Nr przesyłki: {trackingNumbers}</p>
                          </TooltipContent>
                        </Tooltip>
                      );
                    } else if (o.delivery?.smart) {
                      return (
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Truck className="h-4 w-4 mx-auto text-muted-foreground" data-testid={`icon-smart-${order.allegroOrderId}`} />
                          </TooltipTrigger>
                          <TooltipContent>
                            <p>Smart! - {o.delivery?.method?.name || "Allegro Delivery"}</p>
                          </TooltipContent>
                        </Tooltip>
                      );
                    } else if (o.delivery?.method?.name) {
                      return (
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Truck className="h-4 w-4 mx-auto text-muted-foreground" />
                          </TooltipTrigger>
                          <TooltipContent>
                            <p>{o.delivery.method.name}</p>
                          </TooltipContent>
                        </Tooltip>
                      );
                    } else {
                      return <span className="text-muted-foreground">—</span>;
                    }
                  })()}
                </TableCell>
                <TableCell className="text-center">
                  {hasMessage ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <MessageSquare className="h-4 w-4 mx-auto text-blue-600" data-testid={`icon-message-${order.allegroOrderId}`} />
                      </TooltipTrigger>
                      <TooltipContent className="max-w-xs">
                        <p className="text-sm">{o.messageToSeller}</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <span className="text-muted-foreground">—</span>
                  )}
                </TableCell>
                <TableCell className="text-sm text-muted-foreground">
                  {format(new Date(o.boughtAt || o.orderDate), "dd MMM yyyy, HH:mm", {
                    locale: pl,
                  })}
                </TableCell>
              </TableRow>
            );
          })}
        </TableBody>
      </Table>
    </div>
  );
}
