import { Settings, Package, CalendarRange, LogOut, User, Users, Key, Webhook, BookOpen, Server, History } from "lucide-react";
import { Link, useLocation } from "wouter";
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarGroupLabel,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarHeader,
  SidebarFooter,
} from "@/components/ui/sidebar";
import { useAuth } from "@/hooks/use-auth";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { canAccessPage, type UserRole } from "@shared/permissions";
import type { Permission } from "@shared/permissions";

const menuItems = [
  {
    title: "Zamówienia",
    url: "/",
    icon: Package,
    testId: "link-orders",
    permission: "view_orders" as Permission,
  },
  {
    title: "Historia Zmian",
    url: "/order-changes",
    icon: History,
    testId: "link-order-changes",
    permission: "view_recent_updates" as Permission,
  },
  {
    title: "Synchronizacja według dat",
    url: "/date-sync",
    icon: CalendarRange,
    testId: "link-date-sync",
    permission: "manage_sync" as Permission,
  },
  {
    title: "Konfiguracja",
    url: "/settings",
    icon: Settings,
    testId: "link-settings",
    permission: "manage_settings" as Permission,
  },
  {
    title: "Użytkownicy",
    url: "/users",
    icon: Users,
    testId: "link-users",
    permission: "manage_users" as Permission,
  },
  {
    title: "Tokeny API",
    url: "/api-tokens",
    icon: Key,
    testId: "link-api-tokens",
    permission: "manage_api_tokens" as Permission,
  },
  {
    title: "Webhooks",
    url: "/webhooks",
    icon: Webhook,
    testId: "link-webhooks",
    permission: "manage_webhooks" as Permission,
  },
  {
    title: "Integracja Odoo",
    url: "/odoo",
    icon: Server,
    testId: "link-odoo",
    permission: "manage_settings" as Permission,
  },
  {
    title: "Dokumentacja API",
    url: "/api-docs",
    icon: BookOpen,
    testId: "link-api-docs",
    permission: "manage_api_tokens" as Permission,
    external: true, // Oznacz jako external aby otworzyć w nowej zakładce
  },
];

export function AppSidebar() {
  const [location] = useLocation();
  const { user, logoutMutation } = useAuth();

  const handleLogout = () => {
    logoutMutation.mutate();
  };

  const getUserInitials = () => {
    if (user?.firstName && user?.lastName) {
      return `${user.firstName[0]}${user.lastName[0]}`.toUpperCase();
    }
    if (user?.username) {
      return user.username.slice(0, 2).toUpperCase();
    }
    return "U";
  };

  // Filter menu items based on user role
  const visibleMenuItems = menuItems.filter(item => 
    user?.role && canAccessPage(user.role as UserRole, item.url)
  );

  return (
    <Sidebar>
      <SidebarHeader className="p-4">
        <div className="flex items-center gap-3">
          <img 
            src="/alpma-logo.png" 
            alt="Alpma Logo" 
            className="h-8 w-8 object-contain"
          />
          <div>
            <h1 className="font-semibold text-sm">Alpma OMS</h1>
            <p className="text-xs text-muted-foreground">Zarządzanie zamówieniami</p>
          </div>
        </div>
      </SidebarHeader>
      <SidebarContent>
        <SidebarGroup>
          <SidebarGroupLabel>Menu</SidebarGroupLabel>
          <SidebarGroupContent>
            <SidebarMenu>
              {visibleMenuItems.map((item) => (
                <SidebarMenuItem key={item.title}>
                  <SidebarMenuButton
                    asChild
                    isActive={location === item.url}
                    data-testid={item.testId}
                  >
                    {(item as any).external ? (
                      <a href={item.url}>
                        <item.icon className="h-4 w-4" />
                        <span>{item.title}</span>
                      </a>
                    ) : (
                      <Link href={item.url}>
                        <item.icon className="h-4 w-4" />
                        <span>{item.title}</span>
                      </Link>
                    )}
                  </SidebarMenuButton>
                </SidebarMenuItem>
              ))}
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>
      <SidebarFooter className="p-4 border-t">
        {user && (
          <div className="space-y-3">
            <div className="flex items-center gap-3">
              <Avatar className="h-8 w-8">
                <AvatarFallback className="bg-primary/10 text-primary">
                  {getUserInitials()}
                </AvatarFallback>
              </Avatar>
              <div className="flex-1 min-w-0">
                <p className="text-sm font-medium truncate" data-testid="text-username">
                  {user.firstName && user.lastName 
                    ? `${user.firstName} ${user.lastName}`
                    : user.username
                  }
                </p>
                <p className="text-xs text-muted-foreground truncate" data-testid="text-email">
                  {user.email}
                </p>
              </div>
            </div>
            <Button
              variant="outline"
              size="sm"
              className="w-full"
              onClick={handleLogout}
              disabled={logoutMutation.isPending}
              data-testid="button-logout"
            >
              <LogOut className="h-4 w-4 mr-2" />
              {logoutMutation.isPending ? "Wylogowywanie..." : "Wyloguj się"}
            </Button>
          </div>
        )}
        <p className="text-xs text-muted-foreground text-center mt-3">
          © 2025 Alpma OMS
        </p>
      </SidebarFooter>
    </Sidebar>
  );
}
