import { Switch, Route } from "wouter";
import { queryClient } from "./lib/queryClient";
import { QueryClientProvider } from "@tanstack/react-query";
import { Toaster } from "@/components/ui/toaster";
import { TooltipProvider } from "@/components/ui/tooltip";
import { SidebarProvider, SidebarTrigger } from "@/components/ui/sidebar";
import { AppSidebar } from "@/components/app-sidebar";
import { ThemeProvider } from "@/components/theme-provider";
import { AuthProvider, useAuth } from "@/hooks/use-auth";
import { ProtectedRoute } from "@/lib/protected-route";
import Dashboard from "@/pages/dashboard";
import Settings from "@/pages/settings";
import OrderDetails from "@/pages/order-details";
import OrderChanges from "@/pages/order-changes";
import DateSync from "@/pages/date-sync";
import UsersPage from "@/pages/users";
import ApiTokensPage from "@/pages/api-tokens";
import ApiTokenLogsPage from "@/pages/api-token-logs";
import WebhooksPage from "@/pages/webhooks";
import WebhookLogsPage from "@/pages/webhook-logs";
import OdooConfig from "@/pages/odoo-config";
import AuthPage from "@/pages/auth-page";
import ResetPasswordPage from "@/pages/reset-password-page";
import NotFound from "@/pages/not-found";

function Router() {
  const { user, isLoading } = useAuth();

  // Show auth page if not logged in
  if (!isLoading && !user) {
    return (
      <Switch>
        <Route path="/auth" component={AuthPage} />
        <Route path="/reset-password" component={ResetPasswordPage} />
        <Route path="*">
          {() => {
            window.location.href = "/auth";
            return null;
          }}
        </Route>
      </Switch>
    );
  }

  // Show protected routes if logged in
  return (
    <Switch>
      <ProtectedRoute path="/" component={Dashboard} />
      <ProtectedRoute path="/order-changes" component={OrderChanges} />
      <ProtectedRoute path="/date-sync" component={DateSync} />
      <ProtectedRoute path="/settings" component={Settings} />
      <ProtectedRoute path="/users" component={UsersPage} />
      <ProtectedRoute path="/api-tokens" component={ApiTokensPage} />
      <ProtectedRoute path="/api-tokens/logs" component={ApiTokenLogsPage} />
      <ProtectedRoute path="/webhooks" component={WebhooksPage} />
      <ProtectedRoute path="/webhooks/logs" component={WebhookLogsPage} />
      <ProtectedRoute path="/odoo" component={OdooConfig} />
      <ProtectedRoute path="/order/:orderNumber" component={OrderDetails} />
      <Route path="/auth" component={AuthPage} />
      <Route path="/reset-password" component={ResetPasswordPage} />
      <Route component={NotFound} />
    </Switch>
  );
}

function AppContent() {
  const { user, isLoading } = useAuth();
  
  const style = {
    "--sidebar-width": "280px",
    "--sidebar-width-icon": "4rem",
  };

  // Show layout without sidebar during loading or for unauthenticated users
  if (isLoading || !user) {
    return (
      <main className="min-h-screen">
        <Router />
      </main>
    );
  }

  // Show layout with sidebar only for authenticated users
  return (
    <SidebarProvider style={style as React.CSSProperties}>
      <div className="flex h-screen w-full">
        <AppSidebar />
        <div className="flex flex-col flex-1 min-h-0">
          <header className="flex items-center p-4 border-b flex-shrink-0">
            <SidebarTrigger data-testid="button-sidebar-toggle" />
          </header>
          <main className="flex-1 min-h-0">
            <Router />
          </main>
        </div>
      </div>
    </SidebarProvider>
  );
}

export default function App() {
  return (
    <QueryClientProvider client={queryClient}>
      <AuthProvider>
        <TooltipProvider>
          <ThemeProvider defaultTheme="dark">
            <AppContent />
            <Toaster />
          </ThemeProvider>
        </TooltipProvider>
      </AuthProvider>
    </QueryClientProvider>
  );
}
