import functools
import logging
import time
from base64 import b64encode
from copy import copy
from datetime import datetime
from json import JSONDecodeError
from typing import Any, Iterator, Optional, Union
from urllib.parse import urlencode

import requests
from odoo import _
from odoo.addons.trilab_market_base.models.exceptions import MarketError
from odoo.addons.trilab_market_base.models.utils import StrEnum
from requests.auth import AuthBase, HTTPBasicAuth

_logger = logging.getLogger(__name__)

__all__ = [
    'AllegroClient',
    'AllegroError',
    'AllegroMessageStatus',
    'AllegroMessage',
    'AllegroOfferEvent',
    'AllegroOrderEvent',
    'AllegroOrderState',
    'AllegroPayment',
    'AllegroPublicationStatus',
    'AllegroSellMethod',
    'MarketError',
]


class AllegroPayment(StrEnum):
    CASH_ON_DELIVERY = 'CASH_ON_DELIVERY'
    WIRE_TRANSFER = 'WIRE_TRANSFER'
    ONLINE = 'ONLINE'
    SPLIT_PAYMENT = 'SPLIT_PAYMENT'
    EXTENDED_TERM = 'EXTENDED_TERM'


class AllegroOrderState(StrEnum):
    NEW = 'NEW'
    PROCESSING = 'PROCESSING'
    READY_FOR_SHIPMENT = 'READY_FOR_SHIPMENT'
    SENT = 'SENT'
    CANCELLED = 'CANCELLED'

    @classmethod
    def _missing_(cls, value):
        return cls.NEW


class AllegroPublicationStatus(StrEnum):
    INACTIVE = 'INACTIVE'
    ACTIVE = 'ACTIVE'
    ACTIVATING = 'ACTIVATING'
    ENDED = 'ENDED'


class AllegroSellMethod(StrEnum):
    BUY_NOW = 'BUY_NOW'
    AUCTION = 'AUCTION'
    ADVERTISEMENT = 'ADVERTISEMENT'


class AllegroMessageStatus(StrEnum):
    BLOCKED = 'BLOCKED'
    DELIVERED = 'DELIVERED'
    DISMISSED = 'DISMISSED'
    INTERACTING = 'INTERACTING'
    VERIFYING = 'VERIFYING'


class AllegroMessage(StrEnum):
    ASK_QUESTION = 'ASK_QUESTION'
    MAIL = 'MAIL'
    MESSAGE_CENTER = 'MESSAGE_CENTER'


class AllegroOrderEvent(StrEnum):
    """
    BOUGHT: purchase without a checkout form filled in
    FILLED_IN: checkout form filled in but payment is not completed yet so data could still change
    READY_FOR_PROCESSING: payment completed, purchase is ready for processing
    BUYER_CANCELLED: purchase was canceled by buyer
    FULFILLMENT_STATUS_CHANGED: fulfillment status changed
    AUTO_CANCELLED: purchase was canceled automatically by Allegro.
    """

    BOUGHT = 'BOUGHT'
    FILLED_IN = 'FILLED_IN'
    READY_FOR_PROCESSING = 'READY_FOR_PROCESSING'
    BUYER_CANCELLED = 'BUYER_CANCELLED'
    FULFILLMENT_STATUS_CHANGED = 'FULFILLMENT_STATUS_CHANGED'
    AUTO_CANCELLED = 'AUTO_CANCELLED'


class AllegroOfferEvent(StrEnum):
    """
    OFFER_ACTIVATED - offer is visible on site and available for purchase,
                      occurs when offer status changes from ACTIVATING to ACTIVE.
    OFFER_CHANGED - occurs when offer's fields has been changed e.g. description or photos.
    OFFER_ENDED - offer is no longer available for purchase, occurs when offer status changes from ACTIVE to ENDED.
    OFFER_STOCK_CHANGED - stock in an offer was changed either via purchase or by seller.
    OFFER_PRICE_CHANGED - occurs when the price in an offer was changed.
    OFFER_ARCHIVED - offer is no longer available on listing and has been archived.
    OFFER_BID_PLACED - bid was placed on the offer.
    OFFER_BID_CANCELED - bid for offer was canceled.
    OFFER_TRANSLATION_UPDATED - translation of offer updated.
    OFFER_VISIBILITY_CHANGED - visibility of offer changed on marketplaces.
    """

    OFFER_ACTIVATED = 'OFFER_ACTIVATED'
    OFFER_CHANGED = 'OFFER_CHANGED'
    OFFER_ENDED = 'OFFER_ENDED'
    OFFER_STOCK_CHANGED = 'OFFER_STOCK_CHANGED'
    OFFER_PRICE_CHANGED = 'OFFER_PRICE_CHANGED'
    OFFER_ARCHIVED = 'OFFER_ARCHIVED'
    OFFER_BID_PLACED = 'OFFER_BID_PLACED'
    OFFER_BID_CANCELED = 'OFFER_BID_CANCELED'
    OFFER_TRANSLATION_UPDATED = 'OFFER_TRANSLATION_UPDATED'
    OFFER_VISIBILITY_CHANGED = 'OFFER_VISIBILITY_CHANGED'


JSON_CONTENT_TYPES = [
    'application/json',
    'application/vnd.allegro.public.v1+json',
]


class AllegroError(MarketError):
    """
    Allegro Exception
    """

    prefix = 'Allegro API Error:'

    def __init__(self, message: str = None, data: dict | list = None):
        if isinstance(data, dict):
            data = [data]

        self.data = data

        super().__init__(message)

    @staticmethod
    def extract_description(data: list = None) -> str | None:
        if data:
            msgs = []
            for error in data:
                for key in ['userMessage', 'message', 'error_description', 'error', 'code']:
                    if msg := error.get(key):
                        if path := error.get('path'):
                            msg = f'{path}: {msg}'
                        msgs.append(msg)
                        break

            return '\n'.join(msgs)

        return None

    def _parse_error(self) -> str | None:
        if message := self.extract_description(self.data):
            return message

        return super()._parse_error()

    @property
    def description(self):
        desc = None

        if self.data:
            desc = self.extract_description(self.data)

        if not desc and self.message:
            desc = self.message

        if not desc:
            desc = _('unknown error')

        return desc


class AllegroTokenExpired(AllegroError):
    pass


class AllegroTokenMissing(AllegroError):
    pass


class HTTPTokenAuth(AuthBase):
    def __init__(self, token: str):
        self.token = token.strip()

    @staticmethod
    def _get_auth_str(token: str) -> str:
        return f'Bearer {token}'

    def __str__(self):
        return f'token: {self.token[:30]}…{self.token[-30:]}'

    def __repr__(self):
        return f'HTTPTokenAuth(token: {self.__str__()})'

    def __eq__(self, other):
        return self.token == getattr(other, 'token', None)

    def __ne__(self, other):
        return not self == other

    def __call__(self, r):
        r.headers['Authorization'] = self._get_auth_str(self.token)
        return r


class AllegroClient:
    """
    Implementation of Allegro API
    """

    SALES_CENTER_URL = 'https://salescenter.allegro.com'
    AUTH_FIELDS = ['allegro_access_token', 'allegro_access_token_validity', 'allegro_refresh_token', 'state']

    def __init__(self, account_id):
        self.account_id = account_id.exists().ensure_one()
        self.debug_logging = account_id.debug_logging
        self.public_url: str = f'https://{account_id.marketplace_id.base_url}'
        self.api_base_url: str = f'https://api.{account_id.marketplace_id.base_url}'
        self.oauth_base_url: str = f'https://{account_id.marketplace_id.base_url}'
        self.seller_url: str = account_id.marketplace_id.allegro_seller_url
        self.version = account_id.sudo().env.ref('base.module_trilab_allegro').installed_version
        self.lang = self.account_id.marketplace_id.lang_id.code.replace('_', '-')

        if account_id.allegro_environment == 'sandbox':
            self.public_url += '.allegrosandbox.pl'
            self.api_base_url += '.allegrosandbox.pl'
            self.oauth_base_url += '.allegrosandbox.pl'

        self.public_url += f'/{account_id.marketplace_id.allegro_public_url}'

        self.client_id, self.client_secret = account_id.allegro_get_client_data()
        self.auth_token = None

        self.session = requests.Session()

        self.session.headers.update(
            {
                'User-Agent': f'trilab-odoo-allegro/{self.version}',
                'Accept': 'application/vnd.allegro.public.v1+json',
                'Content-Type': 'application/vnd.allegro.public.v1+json',
                'Accept-Language': self.account_id.marketplace_id.lang_id.code.replace('_', '-'),
            }
        )

    def get_api_url(self, *args) -> str:
        return f'{self.api_base_url}/{"/".join(args)}'

    def get_auth_url(self, *args) -> str:
        return f'{self.oauth_base_url}/{"/".join(args)}'

    def get_auction_url(self, *args) -> str:
        return f'{self.public_url}/{"/".join(args)}'

    def get_auction_edit_url(self, *args) -> str:
        if self.account_id.allegro_is_company:
            url = self.SALES_CENTER_URL
        else:
            url = f'{self.account_id.marketplace_id.base_url}'

        if self.account_id.allegro_environment == 'sandbox':
            url += '.allegrosandbox.pl'

        return f'{url}/{"/".join(args)}'

    @staticmethod
    @functools.cache
    def _is_json_ct(content_type: str) -> bool:
        content_type = content_type.split(';')[0].strip()
        return any(ct in content_type for ct in JSON_CONTENT_TYPES)

    def _make_api_request(
        self,
        endpoint: str,
        method: Optional[str] = 'get',
        params: Optional[dict] = None,
        data: Optional[Union[dict, str]] = None,
        json: Optional[dict] = None,
        oauth: Optional[bool] = None,
        headers: Optional[dict] = None,
        raw: Optional[bool] = False,
    ) -> Union[dict[str, Any], list[dict[str, Any]], str, bytes, None]:
        """
        Make an api call, return response
        """

        if not (self.client_id and self.client_secret):
            message = _('ClientID and ClientSecret are required in order to access Allegro REST API.')
            _logger.error(message)
            raise AllegroError(message)

        request: dict[str, Any] = {
            'method': method,
            'headers': copy(headers or {}),
        }

        if params:
            request['params'] = urlencode(params or [], doseq=True)

        if data:
            request['data'] = data

        if json:
            request['json'] = json

        try:
            if oauth:
                request['auth'] = HTTPBasicAuth(self.client_id, self.client_secret)
                request['headers'] |= {'Accept': None, 'Content-Type': None}
                request['url'] = self.get_auth_url(endpoint)
            else:
                if not self.auth_token:
                    self.auth_token = self.get_token()
                request['auth'] = HTTPTokenAuth(self.auth_token)
                request['url'] = self.get_api_url(endpoint)

            # Log the response details for debugging purposes
            _logger.debug(f'{request=}')

            response = self.session.request(**request)

            _logger.debug(
                f'status_code: {response.status_code}, '
                f'response_text: {response.text if self.debug_logging else response.text[:60]}'
            )

            # the response with status code 204 has no content
            if response.ok and not response.content:
                return None

            if raw or not self._is_json_ct(response.headers.get('content-type')):
                return response.content

            response_json = response.json()

            if retry_after := response.headers.get('Retry-After'):
                response_json['retry_after'] = int(retry_after)

            if 'error' in response_json:
                errors = [response_json]

            elif 'errors' in response_json:
                errors = response_json['errors']

            else:
                errors = []

            if errors:
                raise AllegroError(data=errors)

        except requests.RequestException as error:
            if error.response:
                _logger.exception(
                    f'Allegro API error {error.response.status_code}: '
                    f'{error.response.text if self.debug_logging else error.response.text[:60]}'
                )
            else:
                _logger.exception(f'Allegro API error {error}')

            raise AllegroError from error

        except JSONDecodeError as error:
            _logger.error(f'Allegro API error (JSON): {error}')
            raise AllegroError from error

        return response_json

    def get_pairing_data(self) -> dict[str, Any]:
        """
        Client Credentials Flow
        https://developer.allegro.pl/tutorials/uwierzytelnianie-i-autoryzacja-zlq9e75GdIR#device-flow-autoryzacja-uzytkownika
        """

        result = self._make_api_request(
            'auth/oauth/device', method='post', data={'client_id': self.client_id}, oauth=True
        )
        return result

    def get_token(self, *, device_code: Optional[str] = None, raise_exception: bool = True) -> str | None:
        token = data = None

        if device_code:
            data = {'grant_type': 'urn:ietf:params:oauth:grant-type:device_code', 'device_code': device_code}

        else:
            # self.account_id.flush_recordset(self.AUTH_FIELDS)
            self.account_id.fetch(self.AUTH_FIELDS)

            if not self.account_id.allegro_access_token:
                if raise_exception:
                    raise AllegroTokenMissing(_('Missing token or account is not paired!'))

            elif self.account_id.allegro_access_token_validity > time.time():
                token = self.account_id.allegro_access_token

            elif self.account_id.allegro_refresh_token:
                data = {'grant_type': 'refresh_token', 'refresh_token': self.account_id.allegro_refresh_token}

            else:
                if raise_exception:
                    raise AllegroTokenMissing(
                        _('Could not get API token: not in pairing mode and missing refresh token')
                    )

        if token is None and data:
            try:
                result = self._make_api_request('auth/oauth/token', method='post', data=data, oauth=True)

                token = result['access_token']

                self.account_id.allegro_access_token = token
                self.account_id.allegro_access_token_validity = result['expires_in'] + int(time.time())
                self.account_id.allegro_refresh_token = result['refresh_token']
                self.account_id.state = token and 'paired' or 'draft'
                self.account_id.env.cr.commit()

            except (KeyError, AllegroError) as error:
                token = None
                self.account_id.allegro_handle_sync_failure({'error_description': 'invalid_token'})
                self.account_id.env.cr.commit()

                if raise_exception:
                    raise AllegroTokenMissing(_('Could not get API token: %s', str(error)), data=error.data) from error

        return token

    def me(self) -> dict[str, Any]:
        """
        Get basic information about user
        https://developer.allegro.pl/documentation/#operation/meGET
        """

        return self._make_api_request('me')

    def get_marketplaces(self) -> dict[str, Any]:
        """
        Get details for all marketplaces in allegro
        https://developer.allegro.pl/documentation/#operation/marketplacesGET
        """

        return self._make_api_request('marketplaces')

    def get_order_events(
        self,
        *,
        limit: Optional[int] = 100,
        last_event: Optional[str] = None,
        events: Optional[list[AllegroOrderEvent]] = None,
    ) -> list[dict[str, Any]]:
        """
        Get order events
        https://developer.allegro.pl/documentation/#operation/getOrderEventsUsingGET
        """

        params = {'limit': limit}

        if events is None:
            params['type'] = [event for event in AllegroOrderEvent]

        elif events:
            params['type'] = [event for event in events]

        if last_event:
            params['from'] = last_event

        response = self._make_api_request('order/events', params=params)
        return response.get('events', [])

    def get_order_events_statistics(self) -> str:
        """
        Get order events statistics
        https://developer.allegro.pl/documentation/#operation/getOrderEventsStatisticsUsingGET
        """

        response = self._make_api_request('order/event-stats')

        return response.get('latestEvent', {}).get('id')

    def get_offer_events(
        self,
        *,
        limit: Optional[int] = 100,
        last_event: Optional[str] = None,
        events: Optional[list[AllegroOfferEvent]] = None,
    ) -> list[dict[str, Any]]:
        """
        Get events about the seller's offers
        https://developer.allegro.pl/documentation/#operation/getOfferEvents
        """

        params = {'limit': limit}

        if last_event:
            params['from'] = last_event

        if events is None:
            params['type'] = [event for event in AllegroOfferEvent]

        elif events:
            params['type'] = [event for event in events]

        response = self._make_api_request('sale/offer-events', params=params)

        return response.get('offerEvents', [])

    def get_user_orders(
        self,
        *,
        from_date: datetime = None,
        marketplaces: list[str] = None,
        event_type: AllegroOrderEvent = AllegroOrderEvent.READY_FOR_PROCESSING,
        limit: int = 100,
    ) -> dict[str, Any]:
        """
        Get the user's orders
        https://developer.allegro.pl/documentation#operation/getListOfOrdersUsingGET
        """

        params = {'limit': limit, 'status': event_type, 'sort': 'lineItems.boughtAt'}

        if marketplaces:
            params['marketplace.id'] = marketplaces

        if from_date:
            params['lineItems.boughtAt.gte'] = f'{from_date.isoformat(timespec="milliseconds")}Z'

        return self._make_api_request('order/checkout-forms', params=params)

    def get_order_details(self, order_id: str) -> dict[str, Any]:
        """
        Get an order's details
        https://developer.allegro.pl/documentation/#operation/getOrdersDetailsUsingGET
        """

        return self._make_api_request(f'order/checkout-forms/{order_id}')

    def get_product_offer_details(self, ref: str, get_image: Optional[bool] = False) -> dict[str, Any]:
        """
        Get all data of the particular product-offer
        https://developer.allegro.pl/documentation/#operation/getProductOffer
        """

        response = self._make_api_request(f'sale/product-offers/{ref}')

        if get_image and len(response.get('images', [])):
            try:
                response['image'] = b64encode(requests.get(response['images'][0], timeout=5).content)
            except Exception as error:
                _logger.error(f'Exception during requesting an image: {error}')

        return response

    def update_order_status(self, order_id: str, status: AllegroOrderState) -> dict[str, Any]:
        """
        Set seller order status
        https://developer.allegro.pl/documentation/#operation/setOrderFulfillmentUsingPUT
        """

        return self._make_api_request(
            f'order/checkout-forms/{order_id}/fulfillment', method='put', json={'status': status}
        )

    def change_price(self, offer_id: str, command_id: str, data: dict) -> dict[str, Any]:
        """
        Modify the Buy Now price in an offer
        https://developer.allegro.pl/documentation/#operation/createChangePriceCommandUsingPUT
        """

        return self._make_api_request(f'offers/{offer_id}/change-price-commands/{command_id}', method='put', json=data)

    def update_offer(self, offer_id: str, data: dict) -> dict[str, Any]:
        """
        Edit an offer
        https://developer.allegro.pl/documentation/#operation/editProductOffers
        """

        return self._make_api_request(f'sale/product-offers/{offer_id}', method='patch', json=data)

    def get_sellers_offers(
        self,
        *,
        offset: int = 0,
        batch_size: int = 100,
        limit: int = None,
        marketplace: Optional[str] = None,
        status: Optional[list[AllegroPublicationStatus] | AllegroPublicationStatus] = None,
        sell_method: Optional[list[AllegroSellMethod] | AllegroSellMethod] = None,
    ) -> Iterator[dict[str, Any]]:
        """
        Get seller's offers
        https://developer.allegro.pl/documentation/#operation/searchOffersUsingGET
        """

        if limit is not None:
            batch_size = min(batch_size, limit)

        params = {
            'limit': batch_size,
            'offset': offset,
        }

        count = 0

        if marketplace:
            params['publication.marketplace'] = marketplace

        if status:
            if not isinstance(status, list):
                status = [status]
            params['publication.status'] = [_st for _st in status]

        if sell_method:
            if not isinstance(sell_method, list):
                sell_method = [sell_method]
            params['sellingMode.format'] = [_sm for _sm in sell_method]

        while True:
            response = self._make_api_request('sale/offers', params=params)

            for offer in response.get('offers', []):
                yield offer
                count += 1

            if params['offset'] >= response.get('totalCount', 0) or (limit is not None and count >= limit):
                break

            params['offset'] += batch_size

    def _invoice_file_post(self, order_id: str, invoice_id: str, invoice_data: str) -> dict[str, Any]:
        """
        Upload invoice file
        https://developer.allegro.pl/documentation/#operation/uploadOrderInvoiceFile
        """

        return self._make_api_request(
            f'order/checkout-forms/{order_id}/invoices/{invoice_id}/file',
            method='put',
            data=invoice_data,
            headers={'Content-Type': 'application/pdf'},
        )

    def invoice_post(self, order_ref: str, file_name: str, invoice_data: str) -> dict[str, Any]:
        """
        Post new invoice
        https://developer.allegro.pl/documentation/#operation/addOrderInvoicesMetadata
        """

        response = self._make_api_request(
            f'order/checkout-forms/{order_ref}/invoices', method='post', json={'file': {'name': file_name}}
        )

        self._invoice_file_post(order_id=order_ref, invoice_id=response['id'], invoice_data=invoice_data)

        return response.get('id')

    def tracking_number_post(
        self, order_id: str, carrier_tracking_ref: str, carrier_id: Optional[str], carrier_name: Optional[str] = None
    ) -> dict[str, Any]:
        """
        Add a parcel tracking number
        https://developer.allegro.pl/documentation/#operation/createOrderShipmentsUsingPOST
        Carrier name to be provided only if carrierId is OTHER, otherwise it’s ignored. Max 30 characters.
        """

        data = {'carrierId': carrier_id, 'waybill': carrier_tracking_ref}

        if not carrier_id and carrier_name:
            data.update({'carrierId': 'OTHER', 'carrierName': carrier_name[:30]})

        return self._make_api_request(f'order/checkout-forms/{order_id}/shipments', method='post', json=data)

    def get_user_threads(self, offset: Optional[int] = 0, limit: Optional[int] = 20) -> list[dict[str, Any]]:
        """
        list user threads
        https://developer.allegro.pl/documentation/#operation/listThreadsGET
        """

        response = self._make_api_request('messaging/threads', params={'limit': limit, 'offset': offset})
        threads = [thread for thread in response.get('threads', []) if not thread['read']]

        return threads

    def get_messages_thread(
        self, thread: str, offset: Optional[int] = 0, limit: Optional[int] = 20
    ) -> list[dict[str, Any]]:
        """
        list messages in thread
        https://developer.allegro.pl/documentation/#operation/listMessagesGET
        """

        response = self._make_api_request(
            f'messaging/threads/{thread}/messages', params={'limit': limit, 'offset': offset}
        )

        messages = [
            message
            for message in response.get('messages', [])
            if message['status'] == AllegroMessageStatus.DELIVERED
            and message['type'] in (AllegroMessage.ASK_QUESTION, AllegroMessage.MESSAGE_CENTER)
        ]

        return messages

    def post_message_user(self, login: str, text: str) -> dict[str, Any]:
        """
        Write a new message
        https://developer.allegro.pl/documentation/#operation/newMessagePOST
        """

        return self._make_api_request(
            'messaging/messages', method='post', json={'recipient': {'login': login}, 'text': text}
        )

    def post_message_thread(self, thread_id: str, text: str) -> dict[str, Any]:
        """
        Write a new message in thread
        https://developer.allegro.pl/documentation/#operation/newMessageInThreadPOST
        """

        return self._make_api_request(f'messaging/threads/{thread_id}/messages', method='post', json={'text': text})
