import logging
from datetime import datetime
from typing import Any, Optional

import dateutil.parser
from odoo import Command, _, api, fields, models
from odoo.addons.trilab_market_base.models.utils import dictmerge
from odoo.exceptions import UserError
from odoo.tools import float_repr, groupby

from .allegro_client import AllegroClient, AllegroError, AllegroOrderState, MarketError
from .market_mixin import ALLEGRO

_logger = logging.getLogger(__name__)

ALLEGRO_STATUS_NEW = 'NEW'


class Account(models.Model):
    _inherit = 'trilab.market.account'
    _check_company_auto = True

    allegro_environment = fields.Selection(
        [('prod', 'Production'), ('sandbox', 'Sandbox')], string='Environment', default='prod'
    )
    allegro_access_token = fields.Char(string='User Access Token', copy=False)
    allegro_access_token_validity = fields.Integer(string='User Access Token Validity', default=0, copy=False)
    allegro_refresh_token = fields.Char(string='User Refresh Token', copy=False)
    allegro_is_company = fields.Boolean(string='Is Company')
    allegro_restock_auto_renew = fields.Boolean(
        string='Renew Offer on Restock',
        help='Automatically renew expired offer on product restock.',
    )

    # CONSTRAINS

    @api.constrains('sync_payment')
    def _allegro_check_payment_provider(self):
        for account_id in self.ensure_allegro().filtered(
            lambda a_id: not self.env['payment.provider'].search_count(
                [('code', '=', ALLEGRO), ('company_id', '=', a_id.company_id.id)]
            )
        ):
            provider_id = self.env.ref('trilab_allegro.payment_provider_allegro')
            provider_id.sudo().copy(default={'company_id': account_id.company_id.id})

    # noinspection PyUnusedLocal
    @api.constrains('allegro_is_company', 'sync_price')
    def _allegro_check_marketplace_pricelist(self):
        for account_id in self.ensure_allegro().filtered(lambda a_id: a_id.sync_price and not a_id.allegro_is_company):
            raise UserError(_('Price synchronization is available only for commercial (company) accounts.'))

    # TECHNICAL METHODS

    def is_allegro(self):
        # allow to be run against an empty recordset
        return bool(self and self.ensure_one()) and self.market == ALLEGRO

    def ensure_allegro(self, only_paired=True):
        account_ids = self.filtered(lambda a_id: a_id.market == ALLEGRO)

        if only_paired:
            account_ids = account_ids.filtered(lambda a_id: a_id.state == 'paired')

        return account_ids

    def get_client(self, client=None):
        if self.is_allegro():
            return client or AllegroClient(self)

        return super().get_client(client)

    def _get_reset_fields(self):
        return super()._get_reset_fields() + [
            'allegro_access_token',
            'allegro_access_token_validity',
            'allegro_refresh_token',
        ]

    def get_settings(self, *, marketplace_id=None, name: str = None, order_data: dict = None):
        if not marketplace_id and not name and order_data and self.is_allegro():
            name = order_data.get('marketplace', {}).get('id')

        return super().get_settings(marketplace_id=marketplace_id, name=name, order_data=order_data)

    def get_payment_term_mapping(self, order_data: dict):
        if self.is_allegro() and (term_name := (order_data.get('payment') or {}).get('type')):
            return fields.first(self.payment_term_ids.filtered(lambda pt_id: pt_id.allegro_code == term_name))

        return super().get_payment_term_mapping(order_data)

    # noinspection PyMethodMayBeStatic
    def _allegro_post_created_note(self, record_id, api_id=None):
        record_id.message_post(body=_('Created from Allegro transaction %s', api_id or ''))

    # MARKET TECHNICAL METHODS

    # noinspection PyUnusedLocal
    def allegro_handle_sync_failure(self, errors=None):
        if (
            isinstance(errors, dict)
            and errors.get('error_description') == 'invalid_token'
            or isinstance(errors, str)
            and errors == 'invalid_grant'
        ):
            self.allegro_create_pair_activity()
            self.env.cr.commit()

    def allegro_create_pair_activity(self):
        self.ensure_one()

        activity_type_id = self.env.ref('trilab_allegro.allegro_pair_activity')
        model_id = self.env.ref('trilab_market_base.model_trilab_market_account')

        self.env['mail.activity'].search(
            [
                ('res_id', 'in', self.ids),
                ('res_model_id', '=', model_id.id),
                ('activity_type_id', '=', activity_type_id.id),
            ]
        ).unlink()

        self.state = 'draft'

        activity_ids = self.env['mail.activity'].create(
            [
                {
                    'res_id': self.id,
                    'res_model_id': model_id.id,
                    'date_deadline': fields.Date.today(),
                    'summary': 'Allegro',
                    'note': _('Not paired or the token is invalid!'),
                    'activity_type_id': activity_type_id.id,
                    'user_id': user_id.id,
                }
                for user_id in self.user_id | self.env.ref('trilab_allegro.allegro_admin').users
            ]
        )

        return activity_ids

    def allegro_get_client_data(self):
        self.ensure_one()
        icps = self.env['ir.config_parameter'].sudo()
        client_id = icps.get_param(f'allegro_client_id_{self.allegro_environment}')
        client_secret = icps.get_param(f'allegro_client_secret_{self.allegro_environment}')

        return client_id, client_secret

    def allegro_check_account_options(self, client=None):
        self.ensure_allegro()

        try:
            response = self.get_client(client).me()

        except AllegroError as error:
            raise UserError(str(error))

        self.allegro_is_company = (response.get('company') or {}).get('taxId')

        # create missing settings
        if base_marketplace := (response.get('baseMarketplace') or {}).get('id'):
            try:
                # noinspection PyUnboundLocalVariable
                self.get_settings(name=base_marketplace)
            except MarketError:
                marketplace_id = self.env['trilab.market.marketplace'].search(
                    [('market', '=', 'allegro'), ('name', '=', base_marketplace)], limit=1
                )
                self.marketplace_id = marketplace_id.id

                self.settings_ids = [
                    Command.create(
                        {
                            'marketplace_id': marketplace_id.id,
                            'pricelist_id': self.env['product.pricelist'].search([], limit=1, order='id asc').id,
                            'fiscal_position_id': self.env['account.fiscal.position'].search([], limit=1).id,
                        }
                    )
                ]

    @staticmethod
    def _allegro_get_param(param_name: str, parameters: list):
        return next((parameter['values'][0] for parameter in parameters if parameter['id'] == param_name), None)

    # PARTNERS

    def _allegro_parse_address(self, address):
        name = f'{address.get("firstName", "")} {address.get("lastName", "")}'.strip()
        company_name = address.get('companyName')

        country_id = self.env.ref(f'base.{address.get("countryCode").lower()}', raise_if_not_found=False) or self.env[
            'res.lang'
        ].search([('code', '=', address.get('countryCode'))], limit=1)

        result = {
            'name': name or company_name,
            'company_name': company_name,
            'street': address.get('street'),
            'city': address.get('city'),
            'zip': address.get('zipCode', address.get('postCode')),
            'country_id': country_id.id,
            'phone': address.get('phoneNumber'),
        }
        return {key: value for key, value in result.items() if value not in (None, False)}

    def find_or_create_partners(
        self, *, data: dict, client=None
    ) -> tuple[models.BaseModel, models.BaseModel, models.BaseModel]:
        if not self.is_allegro():
            return super().find_or_create_partners(data=data, client=client)

        buyer = data.get('buyer') or {}

        lang_code = None

        if (language := (buyer.get('preferences', {}).get('language', '')).replace('-', '_')) and language in [
            code for code, name in self.env['res.lang'].get_installed()
        ]:
            lang_code = language

        api_ref = buyer.get('id')
        market_ref = self.get_market_ref(api_ref)

        partner_id = self.env['res.partner'].search([('type', '=', 'contact'), ('ref', '=', market_ref)], limit=1)

        if not partner_id:
            first_name = buyer.get('firstName')
            last_name = buyer.get('lastName')

            partner_data = {
                'name': f'{first_name} {last_name}',
                'company_name': buyer.get('companyName'),
                'phone': buyer.get('phoneNumber'),
                'email': buyer.get('email'),
                'ref': market_ref,
                'lang': lang_code,
            }

            if not first_name and not last_name:
                partner_data['name'] = buyer.get('companyName')
                partner_data['company_name'] = None

            partner_id = partner_id.create([{**partner_data, **self._allegro_parse_address(buyer.get('address'))}])
            self._allegro_post_created_note(partner_id, api_ref)

        if invoice_address := (data.get('invoice') or {}).get('address'):
            invoice_partner_id = None
            # noinspection PyUnboundLocalVariable
            invoice_partner_data = self._allegro_parse_address(invoice_address)
            company = invoice_address.get('company') or {}
            partner_vat = partner_registry = None

            for partner_tax in company.get('ids', []):
                tax_type = partner_tax.get('type')
                tax_value = partner_tax.get('value')

                if tax_type in ('CZ_DIC', 'PL_NIP'):
                    partner_vat = tax_value

                elif tax_type == 'CZ_ICO':
                    partner_registry = f'IČO: {tax_value}'

                else:
                    partner_registry = tax_value

            natural_person = invoice_address.get('naturalPerson') or {}

            if natural_person:
                invoice_partner_data['name'] = f'{natural_person.get("firstName")} {natural_person.get("lastName")}'

            else:
                invoice_partner_data['name'] = company.get('name')

            if company:
                invoice_partner_data['company_name'] = company.get('name')

            else:
                invoice_partner_data['company_name'] = None

            if partner_vat or partner_registry:
                if partner_vat:
                    invoice_partner_id = partner_id.search(
                        [('type', 'in', ('contact', 'invoice')), ('vat', '=', partner_vat)], limit=1
                    )

                else:
                    invoice_partner_id = partner_id.search(
                        [('type', 'in', ('contact', 'invoice')), ('company_registry', '=', partner_registry)], limit=1
                    )

                if invoice_partner_id:
                    if natural_person and invoice_partner_id.name != invoice_partner_data['name']:
                        invoice_partner_id.name = invoice_partner_data['name']

                    elif company and invoice_partner_id.company_name != invoice_partner_data['company_name']:
                        invoice_partner_id.company_name = invoice_partner_data['company_name']

            if not invoice_partner_id:
                invoice_partner_id = partner_id.x_allegro_find_existing_address(
                    invoice_partner_data, addr_type='invoice'
                )

            if not invoice_partner_id:
                if partner_id.street:
                    invoice_partner_data.update(
                        {'type': 'invoice', 'lang': lang_code, 'ref': market_ref}
                    )

                    if partner_vat:
                        invoice_partner_data.update({'vat': partner_vat, 'is_company': True})

                    if partner_registry:
                        invoice_partner_data.update({'company_registry': partner_registry})

                    invoice_partner_id = partner_id.create([invoice_partner_data])
                    self._allegro_post_created_note(invoice_partner_id, api_ref)

                else:
                    partner_id.write(invoice_partner_data)
                    invoice_partner_id = partner_id

        else:
            invoice_partner_id = partner_id

        shipping_partner_id = partner_id.browse()

        if address := (data.get('delivery') or {}).get('address'):
            # noinspection PyUnboundLocalVariable
            shipping_data = self._allegro_parse_address(address)

            if shipping_data and 'email' not in shipping_data:
                shipping_data['email'] = buyer.get('email')

            shipping_partner_id = partner_id.x_allegro_find_existing_address(shipping_data, addr_type='delivery')

            if not shipping_partner_id:
                if partner_id.street:
                    shipping_data.update(
                        {'parent_id': partner_id.id, 'type': 'delivery', 'lang': lang_code, 'ref': market_ref}
                    )

                    shipping_partner_id = partner_id.create([shipping_data])
                    self._allegro_post_created_note(shipping_partner_id, api_ref)

                else:
                    partner_id.write(shipping_data)

        if not shipping_partner_id:
            shipping_partner_id = partner_id

        return partner_id, invoice_partner_id, shipping_partner_id

    # PRODUCTS

    def get_product_data(self, *, line_data: dict, client=None) -> dict:
        result = super().get_product_data(line_data=line_data, client=client)

        if not self.is_allegro():
            return result

        data = {
            'name': line_data.get('name'),
            'default_code': (line_data.get('external') or {}).get('id'),
            'barcode': False,
            'image_1920': False,
        }

        if image_data := line_data.get('image'):
            data['image_1920'] = image_data

        icp = self.env['ir.config_parameter'].sudo()
        param_ean_gtin = icp.get_param('allegro_parameter_ean_gtin')
        param_mf_code = icp.get_param('allegro_parameter_mf_code')
        product_set = line_data.get('productSet') or []

        if not data['default_code'] and product_set and len(product_set) > 0:
            data |= {
                'default_code': self._allegro_get_param(param_mf_code, product_set[0]['product']['parameters']),
                'barcode': self._allegro_get_param(param_ean_gtin, product_set[0]['product']['parameters']),
            }

        # if the product name is a digit, the offer does not exist
        if (
            (not data['default_code'] or not data['barcode'])
            and not data['name'].isdigit()
            and (product_details := self.get_client(client).get_product_offer_details(ref=line_data.get('id')))
        ):
            if not data['default_code']:
                data['default_code'] = (product_details.get('external') or {}).get('id')

            if not data['barcode']:
                data['barcode'] = self._allegro_get_param(param_ean_gtin, product_details['parameters'])

            if not data['image_1920']:
                data['image_1920'] = product_details.get('image')

        if not data['default_code']:
            data['default_code'] = line_data.get('id')

        return result | data

    def find_product(self, *, line_data: dict, raise_exception=True, client=None):
        if not self.is_allegro():
            return super().find_product(line_data=line_data, raise_exception=raise_exception, client=client)

        product_data = self.get_product_data(line_data=line_data, client=client)

        return self._find_product(
            offer_ref=self.get_offer_ref(offer_data=line_data),
            name=product_data.get('name'),
            default_code=product_data.get('default_code'),
            barcode=product_data.get('barcode'),
            raise_exception=raise_exception,
            client=client,
        )

    def create_product(self, *, line_data: dict, raise_exception=True, client=None):
        product_id = super().create_product(line_data=line_data, raise_exception=raise_exception, client=client)

        if product_id and self.is_allegro() and line_data:
            self._allegro_post_created_note(product_id, line_data.get('id'))

        return product_id

    def get_shipping_product_data(self, *, delivery_data: dict, client=None) -> dict:
        data = super().get_shipping_product_data(delivery_data=delivery_data, client=client)

        if self.is_allegro():
            data.update({'name': (delivery_data.get('method') or {}).get('name')})

        return data

    def create_shipping_product(self, *, delivery_data: dict, client=None):
        product_id = super().create_shipping_product(delivery_data=delivery_data, client=client)

        if product_id and delivery_data and self.is_allegro():
            self._allegro_post_created_note(product_id, delivery_data.get('id'))

        return product_id

    # OFFERS (auctions)

    def get_offer_ref(self, *, offer_data: dict = None, order_data: dict = None) -> Optional[str]:
        if self.is_allegro():
            if offer_data:
                return offer_data.get('id')

            elif order_data:
                return order_data.get('offer', {}).get('id')

        return super().get_offer_ref(offer_data=offer_data, order_data=order_data)

    def sanitize_offer_data(self, offer_data: dict) -> dict:
        if self.is_allegro():
            # remove binary data
            offer_data.pop('image', None)

        return super().sanitize_offer_data(offer_data)

    def prepare_offer_data(self, offer_data: dict, settings_id, client=None) -> dict:
        result = super().prepare_offer_data(offer_data, settings_id, client=client)

        if self.is_allegro():
            result |= {
                'name': offer_data.get('name'),
                'allegro_republish': self.env['trilab.market.offer'].allegro_get_offer_republish(offer_data, self),
            }

        return result

    def do_sync_offer(
        self, *, ref: str = None, data: dict = None, offer_id=None, update: bool = True, client=None
    ) -> tuple[bool, models.BaseModel]:
        self.ensure_one()

        updated = False

        if not self.is_allegro():
            return super().do_sync_offer(ref=ref, data=data, offer_id=offer_id, update=update, client=client)

        if offer_id is None:
            offer_id = self.env['trilab.market.offer']

        try:
            client = self.get_client(client)

            if not offer_id:
                _logger.debug(f'searching offer for {ref=} {data=}')
                offer_id = self.get_offer(ref=ref, offer_data=data)

                if not offer_id or (update and offer_id.data_changed(data)):
                    # fetch detailed auction info
                    offer_ref = offer_id.ref or ref or self.get_offer_ref(offer_data=data)
                    offer_details = client.get_product_offer_details(offer_ref)

                    if (not offer_id and offer_details) or offer_id.data_changed(offer_details):
                        settings_id = offer_id.settings_id or self.get_settings(
                            name=offer_details.get('publication', {}).get('marketplaces', {}).get('base', {}).get('id')
                        )

                        offer_id = self.update_offer(offer_id=offer_id, data=offer_details, settings_id=settings_id)

                        updated = True
            elif (
                not data
                and update
                and (
                    (offer_details := client.get_product_offer_details(offer_id.ref))
                    and offer_id.data_changed(offer_details)
                )
            ):
                offer_id = self.update_offer(offer_id=offer_id, data=offer_details, settings_id=offer_id.settings_id)
                updated = True

            else:
                offer_id = offer_id or self.env['trilab.market.offer']

            if update and updated and offer_id.state == 'synced':
                updated = False

            if update and offer_id.sync and offer_id.product_id:
                offer_vals = {}

                if self.sync_price:
                    if offer_id.is_system_price_invalid:
                        _logger.debug(
                            f'No pricelist rule found for {offer_id.product_id=} in '
                            f'{offer_id.settings_id.pricelist_id=} skipping...'
                        )

                    elif offer_id.is_price_differ():
                        _logger.debug(
                            f'sync price for offer {offer_id} with {offer_id.offer_price} -> {offer_id.system_price}'
                        )
                        offer_vals['sellingMode'] = {
                            'price': {
                                'amount': float_repr(offer_id.system_price, offer_id.currency_id.decimal_places),
                                'currency': offer_id.currency_id.name,
                            }
                        }

                    else:
                        _logger.debug(f'No pricelist change for {offer_id} skipping...')

                if self.sync_stock:
                    offer_qty, system_qty = offer_id.offer_qty, offer_id.system_qty

                    if offer_qty != system_qty:
                        offer_vals['stock'] = {'available': system_qty}

                        if self.allegro_restock_auto_renew and offer_id.allegro_status == 'ENDED' and system_qty > 0:
                            offer_vals['publication'] = {'status': 'ACTIVE'}

                if offer_id.allegro_republish != offer_id.allegro_get_offer_republish():
                    offer_vals.setdefault('publication', {})
                    offer_vals['publication']['republish'] = offer_id.allegro_republish

                if offer_vals:
                    offer_details = client.update_offer(offer_id.ref, offer_vals)

                    # fetching data here makes no sense, as its update is delayed in allegro
                    # offer_details = client.get_product_offer_details(offer_id.ref)
                    self.update_offer(
                        offer_id=offer_id,
                        data=dictmerge(offer_id.data, offer_details),
                        settings_id=offer_id.settings_id,
                        force=True,
                    )

                    updated = True

            if offer_id.state == 'dirty':
                offer_id.mark_synced()

        except MarketError as error:
            self.log_logging(_('Offer sync exception: %s', str(error)), 'do_sync_offer')
            _logger.exception(f'Synchronization exception: {error}')
            offer_id.mark_error()
            updated = False

        return updated, offer_id

    def do_sync_market_offers(
        self, *, modified: bool = False, offset: int = 0, batch_size: int = 10, limit: int = None
    ):
        seen = []

        if 'lastcall' in self.env.context:
            cron_id = self.env.ref('trilab_allegro.ir_cron_sync_offers')

        else:
            cron_id = self.env['ir.cron']

        for account_id in self.ensure_allegro():
            param_name = f'allegro_sync_offers_offset_{account_id.id}'
            seen.append(param_name)

            page_offset = int(self.env['ir.config_parameter'].sudo().get_param(param_name, offset))

            if page_offset:
                if page_offset == -1:
                    _logger.debug(f'looks like {account_id.name} finished processing, skip it')
                    continue

            _logger.debug(f'starting from offset {page_offset}')

            processed = 0

            try:
                allegro_client = account_id.get_client()
                marketplace_id = fields.first(account_id.settings_ids.filtered('fiscal_position_id').marketplace_id)

                offers = allegro_client.get_sellers_offers(
                    marketplace=marketplace_id and marketplace_id.name or None,
                    offset=page_offset,
                    batch_size=batch_size,
                    limit=limit or batch_size,
                )

                for offer in offers:
                    account_id.do_sync_offer(data=offer, update=not modified, client=allegro_client)
                    processed += 1

            except MarketError as error:
                account_id.log_logging(_('Offer sync exception: %s', str(error)), 'do_sync_offers')
                # raise UserError(_('Synchronization exception: %s', str(error)))

            if processed < (limit or batch_size):
                _logger.debug(f'looks like {account_id.name} finished processing, skip it')
                account_id.offer_sync_date = fields.Datetime.now()
                page_offset = -1

            else:
                page_offset += processed

            self.env['ir.config_parameter'].sudo().set_param(param_name, str(page_offset))

            if page_offset != -1 and cron_id:
                # most probably we have more data and method called from cron, schedule cron again
                _logger.debug(f'reschedule cron again…')
                self._cr.postcommit.add(cron_id._trigger)
                break

        else:
            # finished processing (no rescheduling)

            # cleanup scheduled cron calls
            self.env['ir.cron.trigger'].sudo().search([('cron_id', '=', cron_id.id)]).unlink()

            # cleanup parameters
            for param_name in seen:
                # noinspection PyTypeChecker
                self.env['ir.config_parameter'].sudo().set_param(param_name, None)

            self.env.cr.commit()

        return super().do_sync_market_offers(modified=modified, offset=offset, batch_size=batch_size, limit=limit)

    def do_sync_modified_offers(self, *, batch_size: int = 10, limit: int = None):
        _logger.info(f'allegro do_sync_modified_offers: {batch_size=} {limit=}')
        for settings_id in self.ensure_allegro().settings_ids:
            counter = 0

            try:
                allegro_client = settings_id.account_id.get_client()

                for offer_id in self.env['trilab.market.offer'].search(
                    [('settings_id', '=', settings_id.id), ('sync', '=', True), ('state', 'in', ('draft', 'dirty'))],
                    limit=limit,
                ):
                    settings_id.account_id.do_sync_offer(offer_id=offer_id, client=allegro_client)
                    counter += 1

                    if batch_size is not None and counter >= batch_size:
                        _logger.debug('batch commit…')
                        counter = 0
                        self.env.cr.commit()

            except MarketError as error:
                settings_id.account_id.log_logging(_('Offer sync exception: %s', str(error)), 'do_sync_modified_offers')

        return super().do_sync_modified_offers(batch_size=batch_size, limit=limit)

    def do_check_updated_pricelists(self, batch_size: int = 100):
        changed = super().do_check_updated_pricelists(batch_size)

        if changed and self.ensure_allegro():
            _logger.debug('trigger offers sync')
            self.env.ref('trilab_allegro.ir_cron_sync_modified_offers')._trigger()

        return changed

    # INVENTORY

    def do_sync_stock_picking(self, picking_id, client=None):
        self.ensure_one()

        if self.is_allegro() and self.sync_state:
            client = self.get_client(client)
            try:
                client.update_order_status(picking_id.sale_id.client_order_ref, AllegroOrderState.SENT)
            except MarketError as error:
                msg = _('Could not synchronize stock picking: %s', str(error))
                self.log_logging(msg, 'do_sync_stock_picking')
                raise UserError(msg)

        super().do_sync_stock_picking(picking_id, client=client)

    def do_check_updated_stock(self, batch_size: int = 100):
        changed = super().do_check_updated_stock(batch_size)

        if changed and self.ensure_allegro():
            _logger.debug('trigger offers sync')
            self.env.ref('trilab_allegro.ir_cron_sync_modified_offers')._trigger()

        return changed

    # ORDERS (fulfillment)

    def get_order_ref(self, order_data: dict) -> Optional[str]:
        if self.is_allegro():
            return order_data.get('id')

        return super().get_order_ref(order_data)

    def get_order_date(self, order_data: dict) -> datetime:
        if self.is_allegro():
            return dateutil.parser.parse(order_data['updatedAt'], ignoretz=True)

        return super().get_order_date(order_data)

    def get_order_lines(self, order_data: dict) -> list[dict]:
        result = super().get_order_lines(order_data)

        if self.is_allegro():
            result.extend(order_data.get('lineItems', []))

        return result

    def prepare_sale_order_line_data(
        self, *, order_line_data: dict, offer_id, settings_id, client=None
    ) -> Optional[dict]:
        line = super().prepare_sale_order_line_data(
            order_line_data=order_line_data, offer_id=offer_id, settings_id=settings_id, client=client
        )

        if line is not None and self.is_allegro():
            price_data = order_line_data.get('price') or {}
            transaction_currency_id = self.env.ref(f'base.{price_data.get("currency")}')
            purchase_date = dateutil.parser.parse(order_line_data.get('boughtAt'), ignoretz=True)

            line |= {
                'product_uom_qty': float(order_line_data.get('quantity', 0)),
                'price_unit': transaction_currency_id._convert(
                    float(price_data.get('amount')),
                    settings_id.pricelist_id.currency_id,
                    settings_id.company_id,
                    purchase_date or fields.Datetime.now().date(),
                ),
            }

        return line

    def prepare_sale_order_shipping_data(
        self, *, sale_order_data: dict, order_data: dict, settings_id, client=None
    ) -> dict:
        sale_order_data = super().prepare_sale_order_shipping_data(
            sale_order_data=sale_order_data,
            order_data=order_data,
            settings_id=settings_id,
            client=client,
        )

        if not self.is_allegro():
            return sale_order_data

        delivery_data = order_data.get('delivery')

        if not delivery_data:
            return sale_order_data

        shipping_cost = delivery_data.get('cost') or {}
        shipping_name = (delivery_data.get('method') or {}).get('name')

        shipping_product_id = self.find_or_create_shipping_product(delivery_data=delivery_data, client=client)

        shipping_currency_id = self.env.ref(f'base.{shipping_cost.get("currency")}')

        if dispatch_date := ((delivery_data.get('time') or {}).get('dispatch') or {}).get('to'):
            # noinspection PyUnboundLocalVariable
            sale_order_data['commitment_date'] = dateutil.parser.parse(dispatch_date, ignoretz=True)

        sale_order_data['order_line'].append(
            Command.create(
                {
                    'name': shipping_name,
                    'product_id': shipping_product_id.id,
                    'product_uom_qty': 1.0,
                    'price_unit': shipping_currency_id._convert(
                        float(shipping_cost.get('amount', 0.0)),
                        settings_id.pricelist_id.currency_id,
                        settings_id.company_id,
                        sale_order_data['date_order'],
                    ),
                    'is_delivery': True,
                }
            )
        )

        return sale_order_data

    def process_order_payment(self, *, sale_order_id, order_data: dict, client=None):
        transaction_ids = super().process_order_payment(
            sale_order_id=sale_order_id, order_data=order_data, client=client
        )

        if not self.is_allegro() or not self.sync_payment:
            return transaction_ids

        payment = order_data.get('payment')

        if not payment:
            return transaction_ids

        _logger.debug(f'got payment data for order {sale_order_id.name}: {payment=}')

        payment_ref = payment.get('id')

        if trans_id := transaction_ids.search([('reference', '=', self.get_market_ref(payment_ref))]):
            _logger.debug(f'found existing payment {trans_id}')
            if not trans_id.sale_order_ids & sale_order_id:
                trans_id.sale_order_ids = [Command.link(sale_order_id.id)]
            return transaction_ids | trans_id

        payment_mapping_id = self.get_payment_term_mapping(order_data=order_data)

        if (
            not payment_mapping_id
            or not payment_mapping_id.payment_method_id
            or not payment_mapping_id.payment_provider_id
        ):
            raise UserError(_('Payment terms not configured'))

        if payment_mapping_id.is_cod:
            amt = (order_data.get('summary') or {}).get('totalToPay', {})
        else:
            amt = payment.get('paidAmount') or {}

        amount = float(amt.get('amount') or 0.0)
        currency_id = self.env.ref(f'base.{amt.get("currency")}', raise_if_not_found=False) or sale_order_id.currency_id

        if amount is not None and not currency_id.is_zero(amount):
            transaction_ids += transaction_ids.create(
                {
                    'provider_id': payment_mapping_id.payment_provider_id.id,
                    'payment_method_id': payment_mapping_id.payment_method_id.id,
                    'reference': self.get_market_ref(payment_ref),
                    'provider_reference': payment_ref,
                    'amount': amount,
                    'currency_id': currency_id.id,
                    'partner_id': sale_order_id.partner_id.id,
                    'sale_order_ids': [fields.Command.set(sale_order_id.ids)],
                    'state': 'done',
                }
            )

            _logger.info(f'created payment of {amount} for {sale_order_id.name} ({payment_ref})')

        return transaction_ids

    # noinspection PyUnusedLocal
    def update_sale_order(self, *, sale_order_id, order_data: dict, client=None):
        super().update_sale_order(sale_order_id=sale_order_id, order_data=order_data, client=client)

        if sale_order_id and (event_type := (order_data.get('x_event_data') or {}).get('type')) and event_type in ['AUTO_CANCELLED', 'BUYER_CANCELLED']:
            self.log_logging(f'process order cancel: {sale_order_id.name}', level='info')
            sale_order_id.message_post(body=_('Order cancelled by Allegro (%s)', event_type))

    def create_sale_order(self, *, order_data: dict, client=None):
        sale_order_id = super().create_sale_order(order_data=order_data, client=client)

        if self.is_allegro():
            if message_to_seller := order_data.get('messageToSeller'):
                sale_order_id.message_post(body=message_to_seller, author_id=sale_order_id.partner_id.id)

            if pickup_point := ((order_data.get('delivery') or {}).get('pickupPoint') or {}).get('id'):
                # noinspection PyUnboundLocalVariable
                sale_order_id.message_post(body=_('The buyer chose the following pickup point: %s', pickup_point))

        return sale_order_id

    def process_order_state(self, order_id, order_data: dict = None, client=None):
        if self.is_allegro():
            ref = self.get_order_ref(order_data)
            self.log_logging(f'mark order {ref} as processing', level='info')

            if ((order_data or {}).get('fulfillment') or {}).get('status') == ALLEGRO_STATUS_NEW:
                self.get_client(client).update_order_status(
                    ref or self.get_order_ref(order_data), AllegroOrderState.PROCESSING
                )

        super().process_order_state(order_id=order_id, order_data=order_data, client=client)

    def _allegro_invoice_post(self, order_ref, invoice_id, attachment_data, client=None):
        client = self.get_client(client)

        invoice_number = client.invoice_post(
            order_ref=order_ref, file_name=attachment_data['name'], invoice_data=attachment_data['raw']
        )

        invoice_id.with_context(no_new_invoice=True).message_post(
            body=_('Invoice sent to Allegro (%s).', invoice_number),
            attachments=[(attachment_data['name'], attachment_data['raw'])],
        )

        _logger.info(f'invoice {invoice_id.name} for order {order_ref} sent to Allegro {invoice_number}')

        return invoice_number

    def do_send_invoices(self, invoices_data: dict) -> dict:
        result = super().do_send_invoices(invoices_data)

        for account_id, invoices_group in groupby(
            invoices_data.items(),
            lambda rec_id: fields.first(rec_id[0].line_ids.sale_line_ids.order_id.x_market_account_id),
        ):
            if not account_id.is_allegro() or account_id.state != 'paired' or not account_id.send_invoice:
                continue

            client = account_id.get_client()

            for invoice_id, invoice_data in invoices_group:
                try:
                    attachment_data = invoice_data.get('pdf_attachment_values') or invoice_id.invoice_pdf_report_id

                    for order_ref in invoice_id.line_ids.sale_line_ids.order_id.filtered(
                        lambda o_id: o_id.x_market_account_id.is_allegro() and o_id.client_order_ref
                    ).mapped('client_order_ref'):
                        account_id._allegro_invoice_post(
                            order_ref=order_ref, invoice_id=invoice_id, attachment_data=attachment_data, client=client
                        )
                        # do this once per invoice
                        break

                    else:
                        account_id._allegro_invoice_post(
                            order_ref=invoice_id.ref,
                            invoice_id=invoice_id,
                            attachment_data=attachment_data,
                            client=client,
                        )

                except MarketError as error:
                    account_id.log_logging(_('Invoice send exception: %s', str(error)), 'do_send_invoices')
                    result[invoice_id] = {
                        'error_title': _('Errors when submitting the invoice to allegro:'),
                        'errors': [str(error)],
                    }

        return result

    def do_sync_orders(self):
        for account_id in self.ensure_allegro().filtered('sync_orders'):
            try:
                allegro_client = account_id.get_client()

                response = allegro_client.get_user_orders(from_date=account_id.order_sync_date)

                account_id.log_logging(_('got %s orders', len(response.get('checkoutForms', []))), level='info')

                for order in response.get('checkoutForms', []):
                    account_id.process_order(order)
                    account_id.order_sync_date = account_id.get_order_date(order)

            except MarketError as error:
                msg = _('Order sync exception: %s', str(error))
                _logger.exception(msg)
                account_id.log_logging(msg, level='exception')
                # raise UserError(msg)

        return super().do_sync_orders()

    # EVENTS/WEBHOOKS

    def allegro_do_process_events(
        self, state: str = 'new', from_event: str = None, from_event_id: int = None, batch_size: int = 100
    ):
        search_domain: list[str | tuple[str, str, Any]] = [
            '|',
            ('event_source', '=', 'offer'),
            '&',
            ('event_source', '=', 'order'),
            ('settings_id.account_id.sync_orders', '=', True),
        ]

        if state:
            search_domain.append(('state', '=', state))

        if from_event_id:
            search_domain.append(('id', '>=', from_event_id))

        elif from_event and (event_id := self.env['trilab_allegro.event'].search([('ref', '=', from_event)], limit=1)):
            search_domain.append(('id', '>=', event_id.id))

        self.env['trilab_allegro.event'].search(search_domain).process(batch_size=batch_size)
        self.env.ref('trilab_allegro.ir_cron_sync_modified_offers')._trigger()

    def allegro_do_fetch_offer_events(self, batch_size: int = 100):
        icp = self.env['ir.config_parameter'].sudo()
        event_ids = self.env['trilab_allegro.event']

        for account_id in self.ensure_allegro():
            param_name = f'allegro_last_offer_event_{account_id.id}'

            client = account_id.get_client()
            last_event = icp.get_param(param_name)
            first_event_id = None

            while events := client.get_offer_events(limit=batch_size, last_event=last_event):
                _logger.debug(f'got {len(events)} events, from {last_event=}')
                existing_events = [
                    dta['ref']
                    for dta in event_ids.search_read(
                        [('ref', 'in', [ev['id'] for ev in events]), ('event_source', '=', 'offer')], ['ref']
                    )
                ]
                try:
                    new_event_ids = event_ids.create(
                        [
                            event_ids.parse_allegro_data(event, 'offer', account_id=account_id)
                            for event in events
                            if event['id'] not in existing_events
                        ]
                    )
                except MarketError as error:
                    raise UserError(_('Fetch offer events exception: %s', str(error)))

                if not first_event_id and new_event_ids:
                    first_event_id = fields.first(new_event_ids)

                last_event = events[-1]['id']

                icp.set_param(param_name, last_event)
                self.env.cr.commit()

            if first_event_id:
                self.allegro_do_process_events(from_event_id=first_event_id.id)

    def allegro_do_fetch_order_events(self, batch_size: int = 100):
        icp = self.env['ir.config_parameter'].sudo()
        event_ids = self.env['trilab_allegro.event']

        for account_id in self.ensure_allegro().filtered('sync_orders'):
            param_name = f'allegro_last_order_event_{account_id.id}'

            client = account_id.get_client()
            last_event = icp.get_param(param_name)
            first_event_id = None

            while events := client.get_order_events(limit=batch_size, last_event=last_event):
                _logger.debug(f'got {len(events)} events, from {last_event=}')
                existing_events = [
                    dta['ref']
                    for dta in event_ids.search_read(
                        [('ref', 'in', [ev['id'] for ev in events]), ('event_source', '=', 'order')], ['ref']
                    )
                ]
                try:
                    new_event_ids = event_ids.create(
                        [
                            event_ids.parse_allegro_data(event, 'order', account_id=account_id)
                            for event in events
                            if event['id'] not in existing_events
                        ]
                    )
                except MarketError as error:
                    raise UserError(_('Fetch order events exception: %s, skip event', str(error)))

                if not first_event_id and new_event_ids:
                    first_event_id = fields.first(new_event_ids)

                last_event = events[-1]['id']

                icp.set_param(param_name, last_event)
                self.env.cr.commit()

            if first_event_id:
                self.allegro_do_process_events(from_event_id=first_event_id.id)
